//
// Copyright (c) 2014 Zhugeio. All rights reserved.
//

//   Portions Copyright 2012 Square Inc.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//

#import <Foundation/Foundation.h>
#import <Security/SecCertificate.h>

typedef NS_ENUM(unsigned int, MPWebSocketReadyState) {
    MPWebSocketStateConnecting = 0,
    MPWebSocketStateOpen = 1,
    MPWebSocketStateClosing = 2,
    MPWebSocketStateClosed = 3,
};

@class ZGWebSocket;

extern NSString *const MPWebSocketErrorDomain;

#pragma mark - MPWebSocketDelegate

@protocol MPWebSocketDelegate;

#pragma mark - MPWebSocket

@interface ZGWebSocket : NSObject <NSStreamDelegate>

@property (nonatomic, assign) id <MPWebSocketDelegate> delegate;

@property (nonatomic, readonly) MPWebSocketReadyState readyState;
@property (nonatomic, readonly, retain) NSURL *url;

// This returns the negotiated protocol.
// It will be nil until after the handshake completes.
@property (nonatomic, readonly, copy) NSString *protocol;

// Protocols should be an array of strings that turn into Sec-WebSocket-Protocol.
- (instancetype)initWithURLRequest:(NSURLRequest *)request protocols:(NSArray *)protocols;
- (instancetype)initWithURLRequest:(NSURLRequest *)request;

// Some helper constructors.
- (instancetype)initWithURL:(NSURL *)url protocols:(NSArray *)protocols;
- (instancetype)initWithURL:(NSURL *)url;

// Delegate queue will be dispatch_main_queue by default.
// You cannot set both OperationQueue and dispatch_queue.
- (void)setDelegateOperationQueue:(NSOperationQueue*) queue;
- (void)setDelegateDispatchQueue:(dispatch_queue_t) queue;

// By default, it will schedule itself on +[NSRunLoop mp_networkRunLoop] using defaultModes.
- (void)scheduleInRunLoop:(NSRunLoop *)aRunLoop forMode:(NSString *)mode;
- (void)unscheduleFromRunLoop:(NSRunLoop *)aRunLoop forMode:(NSString *)mode;

// MPWebSockets are intended for one-time-use only.  Open should be called once and only once.
- (void)open;

- (void)close;
- (void)closeWithCode:(NSInteger)code reason:(NSString *)reason;

// Send a UTF8 String or Data.
- (void)send:(id)data;

@end

#pragma mark - MPWebSocketDelegate

@protocol MPWebSocketDelegate <NSObject>

// message will either be an NSString if the server is using text
// or NSData if the server is using binary.
- (void)webSocket:(ZGWebSocket *)webSocket didReceiveMessage:(id)message;

@optional

- (void)webSocketDidOpen:(ZGWebSocket *)webSocket;
- (void)webSocket:(ZGWebSocket *)webSocket didFailWithError:(NSError *)error;
- (void)webSocket:(ZGWebSocket *)webSocket didCloseWithCode:(NSInteger)code reason:(NSString *)reason wasClean:(BOOL)wasClean;

@end

#pragma mark - NSURLRequest (MPCertificateAdditions)

@interface NSURLRequest (MPCertificateAdditions)

@property (nonatomic, retain, readonly) NSArray *mp_SSLPinnedCertificates;

@end

#pragma mark - NSMutableURLRequest (MPCertificateAdditions)

@interface NSMutableURLRequest (MPCertificateAdditions)

@property (nonatomic, retain) NSArray *mp_SSLPinnedCertificates;

@end

#pragma mark - NSRunLoop (SRWebSocket)

@interface NSRunLoop (MPWebSocket)

+ (NSRunLoop *)mp_networkRunLoop;

@end
